// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_CORE_UTIL_SCOPED_PTR_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_CORE_UTIL_SCOPED_PTR_H

#include <cstddef>

/**
 * A pointer that automatically frees the owned memory when it goes out of scope.
 */
template <typename T> class scoped_ptr {
public:
    /**
     * Creates a NULL scoped pointer.
     */
    scoped_ptr() : mPtr(NULL) { }
    /**
     * Creates a scoped pointer which takes ownership of the raw pointer.
     * @param raw The pointer to assume ownership of.
     */
    scoped_ptr(T* raw) : mPtr(raw) { }
    /**
     * Frees the managed pointer.
     */
    ~scoped_ptr() { delete mPtr; }
    /**
     * Takes ownership of the raw pointer. Will free the previous managed pointer if any.
     * @param raw The pointer to take ownership of.
     */
    void operator=(T* raw) {
        delete mPtr;
        mPtr = raw;
    }
    /**
     * Allows dereferencing of the raw pointer.
     * @return The raw managed pointer.
     */
    T* operator->() const {
        return mPtr;
    }
    /**
     * Return the raw pointer. Use with caution.
     * @return The managed pointer.
     */
    T* get() {
        return mPtr;
    }
    /**
     * Test for equality.
     */
    bool operator==(T* raw) {
        return mPtr == raw;
    }
    /**
     * Test for inequality.
     */
    bool operator!=(T* raw) {
        return mPtr != raw;
    }
private:
    T* mPtr;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_CORE_UTIL_SCOPED_PTR_H
